/*-
 * Copyright (c) 2007 Seccuris Inc.
 * All rights reserved.
 *
 * This sofware was developed by Robert N. M. Watson under contract to
 * Seccuris Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD$
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include <net/if.h>
#include "../../src/sys/net/bpf.h"

#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "zbuf_tap.h"

static struct bpf_insn	program[] = {
	{ 0x28, 0, 0, 0x0000000c },
	{ 0x15, 0, 1, 0x00000800 },
	{ 0x6, 0, 0, 0x000005dc },
	{ 0x6, 0, 0, 0x00000000 },
};
static int	program_count = sizeof(program) / sizeof(struct bpf_insn);

int
bpf_ackzbuf(int fd, void *buf, u_int buflen)
{
	struct bpf_zbuf bz;

	bzero(&bz, sizeof(bz));
	bz.bz_bufa = buf;
	bz.bz_buflen = buflen;
	if (ioctl(fd, BIOCACKZBUF, &bz) < 0)
		return (-1);
	return (0);
}

int
bpf_captureall(int fd)
{
	struct bpf_program bp;

	bzero(&bp, sizeof(bp));
	bp.bf_len = program_count;
	bp.bf_insns = program;

	if (ioctl(fd, BIOCSETF, &bp) < 0)
		return (-1);
	return (0);
}

void
bpf_close(int fd)
{

	close(fd);
}

int
bpf_getzmax(int fd, u_int *max)
{

	if (ioctl(fd, BIOCGETZMAX, max) < 0)
		return (-1);
	return (0);
}

int
bpf_getznext(int fd, void **buf, u_int *buflen)
{
	struct bpf_zbuf bz;

	bzero(&bz, sizeof(bz));
	if (ioctl(fd, BIOCGETZNEXT, &bz) < 0)
		return (-1);
	*buf = bz.bz_bufa;
	*buflen = bz.bz_buflen;
	return (0);
}

int
bpf_open(void)
{
	char path[PATH_MAX];
	int fd, i;

	for (i = 0; i < 256; i++) {
		sprintf(path, "/dev/bpf%d", i);
		fd = open(path, O_RDWR);
		if (fd < 0 && errno == EBUSY)
			continue;
		if (fd < 0)
			return (-1);
		break;
	}
	if (i == 256) {
		errno = EBUSY;
		return (-1);
	}
	return(fd);
}

int
bpf_send(int fd, u_char *packet, u_int packetlen)
{
	ssize_t len;

	len = write(fd, packet, packetlen);
	if (len < 0)
		return (0);
	if (len != (ssize_t)packetlen) {
		errno = EMSGSIZE;
		return (-1);
	}
	return (0);
}

int
bpf_setbufmode(int fd, u_int bufmode)
{

	if (ioctl(fd, BIOCSETBUFMODE, &bufmode) < 0)
		return (-1);
	return (0);
}

int
bpf_setif(int fd, const char *ifname)
{
	struct ifreq ifr;

	bzero(&ifr, sizeof(ifr));
	strcpy(ifr.ifr_name, ifname);
	if (ioctl(fd, BIOCSETIF, &ifr) < 0)
		return (-1);
	return (0);
}

int
bpf_setimmediate(int fd, u_int immediate)
{

	if (ioctl(fd, BIOCIMMEDIATE, &immediate) < 0)
		return (-1);
	return (0);
}

int
bpf_setzbuf(int fd, void *bufa, void *bufb, u_int buflen)
{
	struct bpf_zbuf bz;

	bzero(&bz, sizeof(bz));
	bz.bz_bufa = bufa;
	bz.bz_bufb = bufb;
	bz.bz_buflen = buflen;

	if (ioctl(fd, BIOCSETZBUF, &bz) < 0)
		return (-1);
	return (0);
}
