#include <sys/types.h>
#include <sys/queue.h>

#include <dlfcn.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "mac.h"
#include "mac_module.h"

#define	MAC_PARSE_ELEMENT_SEP		','
#define	MAC_PARSE_ELEMENT_SEP_STR	","
#define	MAC_PARSE_POLICY_SEP_STR	"/"
#define	MAC_PARSE_UNKNOWNVALUE		"_unknown"

struct internal_module_entry {
	char		*ime_path;
	void		*ime_handle;

	char		 ime_name[MAC_MAX_POLICY_NAME];

	mm_init		ime_init;
	mm_destroy	ime_destroy;
	mm_free		ime_free;
	mm_from_text	ime_from_text;
	mm_prepare	ime_prepare;
	mm_to_text	ime_to_text;

	LIST_ENTRY(internal_module_entry)	ime_entries;
};

static LIST_HEAD(, internal_module_entry)	internal_module_list;
static int					internal_initialized;

const char *
mac_error(int error)
{

	switch (error) {
	case MAC_SUCCESS:
		return ("Success");
	case MAC_ERROR_ALREADYINIT:
		return ("MAC library already initialized");
	case MAC_ERROR_NOSUCHPOLICY:
		return ("MAC policy not found");
	case MAC_ERROR_NOFROMTEXT:
		return ("MAC policy can't convert text");
	case MAC_ERROR_INVALIDLABELVALUE:
		return ("Invalid label value");
	case MAC_ERROR_POLICYNAMEINVALID:
		return ("Invalid policy name");
	case MAC_ERROR_INSUFFUCIENTRESOURCES:
		return ("Insufficient resources to complete request");
	case MAC_ERROR_NOTTHISMODULE:
		return ("Module does not implement requested policy");
	case MAC_ERROR_NOTSUPPORTED:
		return ("Module does not support requested operation");
	default:
		return ("Unknown error");
	}
}

void
error_to_errno(int error)
{

	switch (error) {
	case MAC_ERROR_INSUFFUCIENTRESOURCES:
		errno = ENOMEM;
	default:
		errno = EINVAL;
	}
}

int
mac_init(void)
{

	return (mac_init_conf(MAC_DEFAULT_CONF));
}

static int
mac_entry_attach(struct internal_module_entry *entry, const char *policyname,
    const char *path, int argc, const char *argv[])
{
	int error;

	if (strlen(policyname)+1 > MAC_MAX_POLICY_NAME)
		return (MAC_ERROR_POLICYNAMEINVALID);

	memset(entry, 0, sizeof(*entry));

	strcpy(entry->ime_name, policyname);
	entry->ime_path = strdup(path);
	if (entry->ime_path == NULL)
		return (MAC_ERROR_INSUFFUCIENTRESOURCES);

	entry->ime_handle = dlopen(entry->ime_path, RTLD_LAZY);
	if (entry->ime_handle == NULL) {
		free(entry->ime_path);
		return (MAC_ERROR_INSUFFUCIENTRESOURCES);
	}

	entry->ime_init = dlsym(entry->ime_handle, MAC_MODULE_INIT);
	entry->ime_destroy = dlsym(entry->ime_handle, MAC_MODULE_DESTROY);
	entry->ime_free = dlsym(entry->ime_handle, MAC_MODULE_FREE);
	entry->ime_from_text = dlsym(entry->ime_handle, MAC_MODULE_FROM_TEXT);
	entry->ime_prepare = dlsym(entry->ime_handle, MAC_MODULE_PREPARE);
	entry->ime_to_text = dlsym(entry->ime_handle, MAC_MODULE_TO_TEXT);

	if (entry->ime_init != NULL) {
		error = entry->ime_init(entry->ime_name, entry->ime_path,
		    argc, argv);
		if (error != MAC_SUCCESS) {
			dlclose(entry->ime_handle);
			free(entry->ime_path);
			return (error);
		}
	}

	return (MAC_SUCCESS);
}

static void
mac_entry_detach(struct internal_module_entry *entry)
{

	if (entry->ime_destroy != NULL)
		entry->ime_destroy();
	dlclose(entry->ime_handle);
	free(entry->ime_path);
	memset(entry, 0, sizeof(*entry));
}

static void
mac_destroy_internal(void)
{
	struct internal_module_entry *entry1, *entry2;

	entry1 = LIST_FIRST(&internal_module_list);
	while (entry1 != NULL) {
		entry2 = LIST_NEXT(entry1, ime_entries);
		LIST_REMOVE(entry1, ime_entries);
		mac_entry_detach(entry1);
		entry1 = entry2;
	}
}

int
mac_init_conf(const char *conffile)
{
	struct internal_module_entry *entry;
	int error;

	error = MAC_SUCCESS;
	if (internal_initialized) {
		error = MAC_ERROR_ALREADYINIT;
		goto just_return;
	}

	LIST_INIT(&internal_module_list);

	entry = (struct internal_module_entry *) malloc(sizeof(*entry));
	if (entry == NULL) {
		error = MAC_ERROR_INSUFFUCIENTRESOURCES;
		goto just_return;
	}

	error = mac_entry_attach(entry, "biba", "./mac_biba.so", 0, NULL);
	if (error != MAC_SUCCESS)
		goto just_return;

	LIST_INSERT_HEAD(&internal_module_list, entry, ime_entries);

	internal_initialized = 1;

just_return:
	if (error != 0)
		mac_destroy_internal();
	return (error);
}

int
mac_destroy(void)
{

	if (!internal_initialized)
		return (MAC_ERROR_NOTINIT);

	mac_destroy_internal();
	return (MAC_SUCCESS);
}

struct internal_module_entry *
mac_module_find(const char *policyname)
{
	struct internal_module_entry *entry;

	LIST_FOREACH(entry, &internal_module_list, ime_entries)
		if (strcmp(entry->ime_name, policyname) == 0)
			return (entry);
	return (NULL);
}

void
mac_free_member(struct mac_member *member)
{
	struct internal_module_entry *entry;

	entry = mac_module_find(member->mm_name);
	if (entry != NULL && entry->ime_free != NULL) {
		entry->ime_free(member);
	} else {
		if (member->mm_data != NULL)
			free(member->mm_data);
	}
}

int
mac_free(struct mac *mac)
{
	int count;

	if (!internal_initialized)
		return (MAC_ERROR_NOTINIT);

	if (mac->m_members != NULL) {
		for (count = 0; count < mac->m_numlivemembers; count++) {
			mac_free_member(&mac->m_members[count]);
		}
		free(mac->m_members);
	}

	free(mac);
	return (MAC_SUCCESS);
}

static struct mac *
mac_alloc(int nummembers)
{
	struct mac_member *members;
	struct mac *mac;

	members = (struct mac_member *) malloc(sizeof(struct mac_member) *
	    nummembers);
	if (members == NULL)
		return (NULL);
	memset(members, 0, sizeof(struct mac_member) * nummembers);

	mac = (struct mac *) malloc(sizeof(*mac));
	if (mac == NULL) {
		free(members);
		return (NULL);
	}
	memset(mac, 0, sizeof(*mac));
	mac->m_nummembers = nummembers;
	mac->m_numlivemembers = 0;
	mac->m_members = members;

	return (mac);
}

int
mac_from_text_np(struct mac **mac, const char *text)
{
	struct mac *temp;
	char *dup, *element, *search;
	int count, error;

	if (!internal_initialized)
		return (MAC_ERROR_NOTINIT);

	dup = strdup(text);
	if (dup == NULL)
		return (MAC_ERROR_INSUFFUCIENTRESOURCES);

	/*
	 * First, count the members to we can allocate a mac_member
	 * array.  Use a simple counting algorithm.
	 */
	count = 1;
	search = dup;
	while (*search != '\0') {
		if (*search == MAC_PARSE_ELEMENT_SEP)
			count++;
		search++;
	}

	temp = mac_alloc(count);
	if (temp == NULL) {
		error = MAC_ERROR_INSUFFUCIENTRESOURCES;
		goto free_dup;
	}

	search = dup;
	while ((element = strsep(&search, MAC_PARSE_ELEMENT_SEP_STR))) {
		struct internal_module_entry *entry;
		char *policyname, *value;

		value = element;
		policyname = strsep(&value, MAC_PARSE_POLICY_SEP_STR);
		if (value == NULL) {
			error = MAC_ERROR_UNPARSEABLELABEL;
			goto free_temp;
		}
		if (strcmp(value, MAC_PARSE_UNKNOWNVALUE) == 0) {
			error = MAC_ERROR_INVALIDLABELVALUE;
			goto free_temp;
		}

		entry = mac_module_find(policyname);
		if (entry == NULL) {
			error = MAC_ERROR_NOSUCHPOLICY;
			goto free_temp;
		}
		strcpy(temp->m_members[temp->m_numlivemembers].mm_name,
		    policyname);
		if (entry->ime_from_text != NULL) {
			error = entry->ime_from_text(
			    &temp->m_members[temp->m_numlivemembers],
			    value);
			if (error != MAC_SUCCESS)
				goto free_temp;
		} else {
			error = MAC_ERROR_NOFROMTEXT;
			goto free_temp;
		}
		temp->m_numlivemembers++;
	}

	goto done;

free_temp:
	mac_free(temp);
free_dup:
	free(dup);
	temp = NULL;
done:
	*mac = temp;
	return (error);
}

struct mac *
mac_from_text(const char *text)
{
	struct mac *mac;
	int error;

	error = mac_from_text_np(&mac, text);
	if (error != MAC_SUCCESS) {
		error_to_errno(error);
		mac = NULL;
	}

	return (mac);
}

int
mac_prepare(struct mac **mac, int numpolicies, const char *policies[])
{
	struct internal_module_entry *entry;
	struct mac *temp;
	int count, error;

	if (!internal_initialized)
		return (MAC_ERROR_NOTINIT);

	temp = mac_alloc(numpolicies);
	if (temp == NULL)
		return (MAC_ERROR_INSUFFUCIENTRESOURCES);

	for (count = 0; count < numpolicies; count++) {
		entry = mac_module_find(policies[count]);
		if (entry == NULL) {
			mac_free(temp);
			*mac = NULL;
			return (MAC_ERROR_NOSUCHPOLICY);
		}
		if (entry->ime_prepare == NULL) {
			mac_free(temp);
			*mac = NULL;
			return (MAC_ERROR_NOTSUPPORTED);
		}
		strcpy(temp->m_members[count].mm_name, policies[count]);
		error = entry->ime_prepare(&temp->m_members[count]);
		if (error) {
			mac_free(temp);
			*mac = NULL;
			return (error);
		}
		temp->m_numlivemembers++;
	}

	*mac = temp;
	return (MAC_SUCCESS);
}

int
mac_to_text_np(struct mac *mac, char **text)
{
	struct internal_module_entry *entry;
	struct mac_member *member;
	char *string, *tempstring, *element, *policyvalue;
	int error, i;

	if (!internal_initialized)
		return (MAC_ERROR_NOTINIT);

	element = NULL;
	string = NULL;
	for (i = 0; i < mac->m_numlivemembers; i++) {
		member = &mac->m_members[i];
		entry = mac_module_find(member->mm_name);
		if (entry == NULL)
			element = strdup(MAC_PARSE_UNKNOWNVALUE);
		else if (entry->ime_to_text == NULL)
			element = strdup(MAC_PARSE_UNKNOWNVALUE);
		else {
			error = entry->ime_to_text(member, &policyvalue);
			if (error != MAC_SUCCESS)
				goto error_handler;

			asprintf(&element, "%s%s%s", member->mm_name,
			    MAC_PARSE_POLICY_SEP_STR, policyvalue);
			free(policyvalue);
		}
		if (element == NULL) {
			error = MAC_ERROR_INSUFFUCIENTRESOURCES;
			goto error_handler;
		}

		if (string == NULL) {
			string = element;
		} else {
			tempstring = string;
			asprintf(&string, "%s,%s", tempstring, element);
			free(tempstring);
			free(element);
			element = NULL;
		}
	}

	*text = string;
	return (MAC_SUCCESS);

error_handler:
	if (string != NULL)
		free(string);
	if (element != NULL)
		free(element);

	return (error);
}

char *
mac_to_text(struct mac *mac, size_t *len_p)
{
	char *string;
	int error;

	error = mac_to_text_np(mac, &string);
	if (error != MAC_SUCCESS) {
		string = NULL;
		error_to_errno(error);
	} else {
		if (len_p != NULL)
			*len_p = strlen(string);
	}
	return (string);
}
