/*-
 * Copyright (c) 2004 Networks Associates Technology, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <libnfs.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>

#define	SMALLBUF	128

int
main(int argc, char *argv[])
{
	struct nfsnode *rootnode, *tmpnode;
	struct nfsmount *nfsm;
	struct nfsinfo info;
	char buf[SMALLBUF];
	struct nfscred nc;
	struct stat sb;
	ssize_t len;
	
	/*
	 * Create a root credentil using uid 0, gid 0, and no additional
	 * groups.  Many NFS server implementations will ignore mount
	 * requests from NULL or other credentials.
	 */
	bzero(&nc, sizeof(nc));
	if (nfs_mount(&nfsm, &nc, "127.0.0.1", "/test") == -1) {
		perror("nfs_mount");
		return (-1);
	}

	/*
	 * Print a few stats to illustrate that mounting pulled them down
	 * successfully.
	 */
	if (nfs_fsinfo(nfsm, &info) == -1) {
		perror("nfs_fsinfo");
		goto out;
	}

	printf("Successfully mounted 127.0.0.1:/test\n");
	printf("\n");
	printf("rtmax: %u     rtpref: %u     rtmult: %u\n", info.ni_rtmax,
	    info.ni_rtpref, info.ni_rtmult);
	printf("wtmax: %u     wtpref: %u     wtmult: %u\n", info.ni_wtmax,
	    info.ni_wtpref, info.ni_wtmult);
	printf("dtpref: %u     maxfilesize: %llu\n", info.ni_dtpref,
	    info.ni_maxfilesize);
	printf("\n");

	if (nfs_null(nfsm) == -1) {
		perror("nfs_null");
		goto out;
	}

	if (nfs_getroot(nfsm, &rootnode) == -1) {
		perror("nfs_getroot");
		goto out;
	}
	printf("Got root!\n");

	if (nfs_getattr(nfsm, rootnode, &sb) == -1) {
		perror("nfs_getattr");
		goto out_deref_root;
	}

	printf("Root size is %llu\n", sb.st_size);

	if (nfs_lookup(nfsm, rootnode, ".", &tmpnode) == -1) {
		perror("nfs_lookup: .");
		goto out_deref_root;
	}

	if (tmpnode != rootnode)
		printf("/. != /\n");
	else
		printf("/. == /\n");

	nfs_node_deref(tmpnode);

	if (nfs_access(nfsm, rootnode, R_OK) == -1)
		perror("nfs_access: .");
	else
		printf("Read access on root OK\n");

	if (nfs_create(nfsm, rootnode, "test", 0644, &tmpnode) == -1) {
		perror("nfs_create(test)");
		goto out_deref_root;
	}

	len = nfs_read(nfsm, tmpnode, 0, buf, SMALLBUF);
	if (len != -1) {
		printf("nfs_read(tmpnode) returned %d bytes\n", len);
	} else
		perror("Read on /test/test node");

	memset(buf, 'A', SMALLBUF);
	len = nfs_write(nfsm, tmpnode, 0, buf, SMALLBUF);
	if (len != -1) {
		printf("nfs_write(tmpnode) returned %d bytes\n", len);
	} else
		perror("Write on /test/test node");

	if (nfs_chmod(nfsm, tmpnode, 0775) == -1)
		perror("nfs_chmod");

	if (nfs_chown(nfsm, tmpnode, 0, 5) == -1)
		perror("nfs_chown");

	if (nfs_truncate(nfsm, tmpnode, 0) == -1)
		perror("nfs_truncate");

	nc.nc_uid = 5;
	if (nfs_setcred(nfsm, &nc) == -1) {
		perror("nfs_setcred");
		nfs_node_deref(tmpnode);
		goto out_deref_root;
	}

	memset(buf, 'B', SMALLBUF);
	len = nfs_write(nfsm, tmpnode, 0, buf, SMALLBUF);
	if (len != -1) {
		printf("nfs_write(tmpnode) as 5 returned %d bytes "
		    "(failure undesired)\n", len);
	} else
		perror("write on /test/test node as 5 (failure desired)");

	nc.nc_gid = 5;
	if (nfs_setcred(nfsm, &nc) == -1) {
		perror("nfs_setcred");
		nfs_node_deref(tmpnode);
		goto out_deref_root;
	}

	memset(buf, 'B', SMALLBUF);
	len = nfs_write(nfsm, tmpnode, 0, buf, SMALLBUF);
	if (len != -1) {
		printf("nfs_write(tmpnode) as 5 returned %d bytes "
		    "(success undesired)\n", len);
	} else
		perror("write on /test/test node as 5 (success desired)");
	
	nfs_node_deref(tmpnode);

	nc.nc_uid = 0;
	nc.nc_gid = 0;
	if (nfs_setcred(nfsm, &nc) == -1) {
		perror("nfs_setcred");
		goto out_deref_root;
	}

	if (nfs_mkdir(nfsm, rootnode, "test2", 0755, &tmpnode) == -1) {
		perror("nfs_mkdir");
		goto out_deref_root;
	}

	nfs_node_deref(tmpnode);

out_deref_root:
	nfs_node_deref(rootnode);
out:
	nfs_unmount(nfsm);
	return (0);
}
